# Copyright 2014-2020 The ODL contributors
#
# This file is part of ODL.
#
# This Source Code Form is subject to the terms of the Mozilla Public License,
# v. 2.0. If a copy of the MPL was not distributed with this file, You can
# obtain one at https://mozilla.org/MPL/2.0/.

"""Tomographic datasets from Mayo Clinic.

In addition to the standard ODL requirements, this library also requires:

    - tqdm
    - dicom
    - A copy of the Mayo dataset, see
    https://www.aapm.org/GrandChallenge/LowDoseCT/#registration
"""

from __future__ import division
import numpy as np
import os
import sys
import pydicom as dicom
import odl
import tqdm
import math
import cv2
from odl.discr import grid
from sklearn.neighbors import KDTree

from pydicom.datadict import DicomDictionary #, NameDict, CleanName
from odl.discr.discr_utils import linear_interpolator
from odl.contrib.datasets.ct.mayo_dicom_dict import new_dict_items
from scipy.spatial.transform import Rotation
import torch
import torch.nn.functional as F
from skimage.metrics import structural_similarity as compare_ssim
# Update the DICOM dictionary with the extra Mayo tags
DicomDictionary.update(new_dict_items)
#NameDict.update((CleanName(tag), tag) for tag in new_dict_items)


__all__ = ('load_projections', 'load_reconstruction')



def convert_to_quaternion(image_orientation_patient):
  """
  Converts image_orientation_patient data (6 elements) to a quaternion.

  Args:
    image_orientation_patient: A list containing 6 direction cosines.

  Returns:
    A list representing the quaternion (w, x, y, z).
  """

  Rx, Ry, Rz, Cx, Cy, Cz = image_orientation_patient

  a = (Rx + Cx) / 2
  b = (Ry + Cy) / 2
  c = (Rz + Cz) / 2
  d = (Rx - Cx) / 2
  e = (Ry - Cy) / 2
  f = (Rz - Cz) / 2

  largest = max(a, b, c, d, e, f)

  if largest == a or largest == b or largest == c:
    w = math.sqrt(1 + a - b - c)
    x = d / (2 * w)
    y = e / (2 * w)
    z = f / (2 * w)
  elif largest == d:
    w = math.sqrt(1 + d - a - b)
    x = 2 * w
    y = e / (2 * w)
    z = f / (2 * w)
  elif largest == e:
    w = math.sqrt(1 + e - a - b)
    x = d / (2 * w)
    y = 2 * w
    z = f / (2 * w)
  else:
    w = math.sqrt(1 + f - a - b)
    x = d / (2 * w)
    y = e / (2 * w)
    z = 2 * w

  return [w, x, y, z]


def transform(image_position_patient, image_orientation_patient, PixelSpacing):
    transformation_matrix = np.zeros((4, 4))
    transformation_matrix[0, 0:3] = image_orientation_patient[0:3] * PixelSpacing[0]
    transformation_matrix[1, 0:3] = image_orientation_patient[3:6] * PixelSpacing[1]
    transformation_matrix[0:3, 3] = image_position_patient
    transformation_matrix[2, 2] = 1
    transformation_matrix[3, 3] = 1
    rotation_matrix = transformation_matrix[0:3, 0:3]
    translation_vector = transformation_matrix[0:3, 3]

    # Convert the rotation matrix to a quaternion
    rotation = Rotation.from_matrix(rotation_matrix)
    quaternion = rotation.as_quat()

    # Invert the pose
    quaternion = -quaternion
    translation_vector = -translation_vector
    print(image_position_patient, translation_vector)
    return quaternion, translation_vector


def _read_projections_alternative(folder, indices):
    """Read mayo projections from a folder by wsking 2024.3.4
    following the DICOM-CT-PD User Manual Version 3""" 
    datasets = []

    # Get the relevant file names
    file_names = sorted([f for f in os.listdir(folder) if f.endswith(".dcm")])

    if len(file_names) == 0:
        raise ValueError('No DICOM files found in {}'.format(folder))

    if indices is not None:
        file_names = file_names[indices]

    data_array = None

    for i, file_name in enumerate(tqdm.tqdm(file_names,
                                            'Loading projection data')):
        # read the file
        dataset = dicom.read_file(folder + '/' + file_name)
        dataset.NumberofDetectorRows = dataset[0x70291010].value
        dataset.NumberofDetectorColumns = dataset[0x70291011].value
        dataset.HUCalibrationFactor = float(dataset[0x70411001].value)
        dataset.DetectorFocalCenterAngularPosition = dataset[0x70311001].value
        dataset.DetectorElementTransverseSpacing = dataset[0x70291002].value
        dataset.DetectorElementAxialSpacing = dataset[0x70291006].value
        dataset.DetectorCentralElement = dataset[0x70311033].value
        dataset.DetectorFocalCenterRadialDistance = dataset[0x70311003].value
        dataset.ConstantRadialDistance = dataset[0x70311031].value
        dataset.DetectorFocalCenterAxialPosition = dataset[0x70311002].value
        dataset.SourceAxialPositionShift = dataset[0x7033100C].value # 0x7033100B?
        dataset.SourceAngularPositionShift = dataset[0x7033100B].value
        dataset.SourceRadialDistanceShift = dataset[0x7033100D].value

        # Get some required data
        rows = dataset.NumberofDetectorRows
        cols = dataset.NumberofDetectorColumns
        hu_factor = dataset.HUCalibrationFactor
        rescale_intercept = dataset.RescaleIntercept
        rescale_slope = dataset.RescaleSlope

        # Load the array as bytes
        proj_array = np.array(np.frombuffer(dataset.PixelData, 'H'),
                              dtype='float32')
        proj_array = proj_array.reshape([rows, cols], order='F').T

        # Rescale array
        proj_array *= rescale_slope
        proj_array += rescale_intercept
        proj_array /= hu_factor

        # Store results
        if data_array is None:
            # We need to load the first dataset before we know the shape
            data_array = np.empty((len(file_names), cols, rows),
                                  dtype='float32')

        data_array[i] = proj_array[:, ::-1]
        datasets.append(dataset)

    return datasets, data_array



def load_projections(folder, indices=None):
    """Load geometry and data stored in Mayo format from folder.

    Parameters
    ----------
    folder : str
        Path to the folder where the Mayo DICOM files are stored.
    indices : optional
        Indices of the projections to load.
        Accepts advanced indexing such as slice or list of indices.

    Returns
    -------
    geometry : ConeBeamGeometry
        Geometry corresponding to the Mayo projector.
    proj_data : `numpy.ndarray`
        Projection data, given as the line integral of the linear attenuation
        coefficient (g/cm^3). Its unit is thus g/cm^2.
    """
    datasets, data_array = _read_projections_alternative(folder, indices)

    # Get the angles
    angles = [d.DetectorFocalCenterAngularPosition for d in datasets]
    angles = -np.unwrap(angles) - np.pi  # different definition of angles

    # Set minimum and maximum corners
    shape = np.array([datasets[0].NumberofDetectorColumns,
                      datasets[0].NumberofDetectorRows])
    pixel_size = np.array([datasets[0].DetectorElementTransverseSpacing,
                           datasets[0].DetectorElementAxialSpacing])

    # Correct from center of pixel to corner of pixel
    minp = -(np.array(datasets[0].DetectorCentralElement) - 0.5) * pixel_size
    maxp = minp + shape * pixel_size

    # Select geometry parameters
    src_radius = datasets[0].DetectorFocalCenterRadialDistance
    det_radius = (datasets[0].ConstantRadialDistance -
                  datasets[0].DetectorFocalCenterRadialDistance)

    # For unknown reasons, mayo does not include the tag
    # "TableFeedPerRotation", which is what we want.
    # Instead we manually compute the pitch
    pitch = ((datasets[-1].DetectorFocalCenterAxialPosition -
              datasets[0].DetectorFocalCenterAxialPosition) /
             ((np.max(angles) - np.min(angles)) / (2 * np.pi)))

    # Get flying focal spot data
    offset_axial = np.array([d.SourceAxialPositionShift for d in datasets])
    offset_angular = np.array([d.SourceAngularPositionShift for d in datasets])
    offset_radial = np.array([d.SourceRadialDistanceShift for d in datasets])

    # TODO(adler-j): Implement proper handling of flying focal spot.
    # Currently we do not fully account for it, merely making some "first
    # order corrections" to the detector position and radial offset.

    # Update angles with flying focal spot (in plane direction).
    # This increases the resolution of the reconstructions.
    angles = angles - offset_angular

    # We correct for the mean offset due to the rotated angles, we need to
    # shift the detector.
    offset_detector_by_angles = det_radius * np.mean(offset_angular)
    minp[0] -= offset_detector_by_angles
    maxp[0] -= offset_detector_by_angles

    # We currently apply only the mean of the offsets
    src_radius = src_radius + np.mean(offset_radial)

    # Partially compensate for a movement of the source by moving the object
    # instead. We need to rescale by the magnification to get the correct
    # change in the detector. This approximation is only exactly valid on the
    # axis of rotation.
    mean_offset_along_axis_for_ffz = np.mean(offset_axial) * (
        src_radius / (src_radius + det_radius))

    # Create partition for detector
    detector_partition = odl.uniform_partition(minp, maxp, shape)

    # Convert offset to odl definitions
    offset_along_axis = (mean_offset_along_axis_for_ffz +
                         datasets[0].DetectorFocalCenterAxialPosition -
                         angles[0] / (2 * np.pi) * pitch)

    # Assemble geometry
    angle_partition = odl.nonuniform_partition(angles)
    geometry = odl.tomo.ConeBeamGeometry(angle_partition,
                                         detector_partition,
                                         src_radius=src_radius,
                                         det_radius=det_radius,
                                         pitch=pitch,
                                         offset_along_axis=offset_along_axis)

    # Create a *temporary* ray transform (we need its range)
    spc = odl.uniform_discr([-1] * 3, [1] * 3, [32] * 3)
    ray_trafo = odl.tomo.RayTransform(spc, geometry, interp='linear')

    # convert coordinates
    theta, up, vp = ray_trafo.range.grid.meshgrid
    d = src_radius + det_radius
    u = d * np.arctan(up / d)
    v = d / np.sqrt(d**2 + up**2) * vp

    # Calculate projection data in rectangular coordinates since we have no
    # backend that supports cylindrical
    interpolator = linear_interpolator(
        data_array, ray_trafo.range.grid.coord_vectors
    )
    proj_data = interpolator((theta, u, v))

    return geometry, proj_data#.asarray()


def save_ct_images(data,save_root):
    if not os.path.exists(save_root):
        os.makedirs(save_root)
    for i in range(data.shape[2]):
        img = data[:,:,i]
        img = (img - img.min())/(img.max()-img.min()) # range 0-1
        img = (img*255).astype('uint8')
        img = np.rot90(img,1)
        img = img.astype(np.uint8)
        cv2.imwrite(save_root + '/%d.png'%i,img)


def load_npy(root):
    fname_list = os.listdir(root)
    fname_list = sorted(fname_list, key=lambda x: float(x.split(".")[0]))
    print(fname_list)
    all_img = []

    print("Loading all data")
    for fname in tqdm.tqdm(fname_list):
        just_name = fname.split('.')[0]
        img = torch.from_numpy(np.load(os.path.join(root, fname), allow_pickle=True))
        h, w = img.shape
        # img = img.view(1, 1, h, w)
        img = img.view(1, h, w)
        all_img.append(img)
    all_img = torch.cat(all_img, dim=0)
    print(f"Data loaded shape : {all_img.shape}")
    img = all_img.cuda()
    return img


def load_reconstruction(folder, slice_start=0, slice_end=-1, suffix='.IMA'):
    """Load a volume from folder, also returns the corresponding partition.

    Parameters
    ----------
    folder : str
        Path to the folder where the DICOM files are stored.
    slice_start : int
        Index of the first slice to use. Used for subsampling.
    slice_end : int
        Index of the final slice to use.

    Returns
    -------
    partition : `odl.RectPartition`
        Partition describing the geometric positioning of the voxels.
    data : `numpy.ndarray`
        Volumetric data. Scaled such that data = 1 for water (0 HU).

    Notes
    -----
    DICOM data is highly non trivial. Typically, each slice has been computed
    with a slice tickness (e.g. 3mm) but the slice spacing might be
    different from that.

    Further, the coordinates in DICOM is typically the *middle* of the pixel,
    not the corners as in ODL.

    This function should handle all of these peculiarities and give a volume
    with the correct coordinate system attached.
    """
    file_names = sorted([f for f in os.listdir(folder) if f.endswith(suffix)])

    if len(file_names) == 0:
        raise ValueError('No DICOM files found in {}'.format(folder))

    volumes = []
    datasets = []

    # file_names = file_names[slice_start:slice_end]

    for file_name in tqdm.tqdm(file_names, 'loading volume data'):
        # read the file
        dataset = dicom.read_file(folder + '/' + file_name)

        # Get parameters
        pixel_size = np.array(dataset.PixelSpacing)
        pixel_thickness = float(dataset.SliceThickness)
        rows = dataset.Rows
        cols = dataset.Columns

        # Get data array and convert to correct coordinates
        data_array = np.array(np.frombuffer(dataset.PixelData, 'H'),
                              dtype='float32')
        data_array = data_array.reshape([cols, rows], order='C')
        data_array = np.rot90(data_array, -1)

        # Convert from storage type to densities
        # TODO: Optimize these computations
        hu_values = (dataset.RescaleSlope * data_array +
                     dataset.RescaleIntercept)
        densities = (hu_values + 1000) / 1000

        # Store results
        volumes.append(densities)
        datasets.append(dataset)
        # pose = transform(np.array(dataset.ImagePositionPatient), np.array(dataset.ImageOrientationPatient),np.array(dataset.PixelSpacing))

    voxel_size = np.array(list(pixel_size) + [pixel_thickness])
    shape = np.array([rows, cols, len(volumes)])

    # Compute geometry parameters
    mid_pt = (np.array(dataset.ReconstructionTargetCenterPatient) -
              np.array(dataset.DataCollectionCenterPatient))
    reconstruction_size = (voxel_size * shape)
    min_pt = mid_pt - reconstruction_size / 2
    max_pt = mid_pt + reconstruction_size / 2

    # axis 1 has reversed convention
    min_pt[1], max_pt[1] = -max_pt[1], -min_pt[1] # !!!!!!!!!!!!!!!!!!!!!! check this line 2024.3.9

    if len(datasets) > 1:
        slice_distance = np.abs(
            float(datasets[1].DataCollectionCenterPatient[2]) -
            float(datasets[0].DataCollectionCenterPatient[2]))
    else:
        # If we only have one slice, we must approximate the distance.
        slice_distance = pixel_thickness

    # The middle of the minimum/maximum slice can be computed from the
    # DICOM attribute "DataCollectionCenterPatient". Since ODL uses corner
    # points (e.g. edge of volume) we need to add half a voxel thickness to
    # both sides.
    try:
        min_pt[2] = -np.array(datasets[0].DataCollectionCenterPatient)[2]
        min_pt[2] -= 0.5 * slice_distance
        max_pt[2] = -np.array(datasets[-1].DataCollectionCenterPatient)[2]
        max_pt[2] += 0.5 * slice_distance
        partition = odl.uniform_partition(min_pt, max_pt, shape)
    except:
        print('!!!!!!!!Load data in the reverse order!!!!!!!!!!')
        min_pt[2] = -np.array(datasets[-1].DataCollectionCenterPatient)[2]
        min_pt[2] -= 0.5 * slice_distance
        max_pt[2] = -np.array(datasets[0].DataCollectionCenterPatient)[2]
        max_pt[2] += 0.5 * slice_distance
        partition = odl.uniform_partition(min_pt, max_pt, shape)
    

    # volume = np.transpose(np.array(volumes), (1, 2, 0))
    volume = np.array(volumes)
    return partition, volume


def resize_and_crop(image, image_size):
        # Crop slices in z dim
    center_idx = int(image.shape[0] / 2)
    num_slice = int(image_size[0] / 2)
    image = image[center_idx-num_slice:center_idx+num_slice, :, :]
    im_size = image.shape
    print(image.shape, center_idx, num_slice)

    # Complete 3D input image as a squared x-y image
    if not(im_size[1] == im_size[2]):
        length = np.max([im_size[1], im_size[2]])
        zerp_padding = np.zeros([im_size[0], length, length])
        if im_size[1] > im_size[2]:
            zerp_padding[:, :, np.int64((im_size[1]-im_size[2])/2):np.int64((im_size[1]-im_size[2])/2)+im_size[2]] = image
        else:
            zerp_padding[:, np.int64((im_size[2]-im_size[1])/2):np.int64((im_size[2]-im_size[1])/2)+im_size[1], :] = image
        image = zerp_padding
    # Resize image in x-y plane
    image = torch.tensor(image, dtype=torch.float32)[None, ...]  # [B, C, H, W]
    image = F.interpolate(image, size=(image_size[1], image_size[2]), mode='bilinear', align_corners=False)

    # Scaling normalization
    image = image / torch.max(image)  # [B, C, H, W], [0, 1]
    # image = (image-image.min())/(image.max()-image.min())
    image = image.permute(1, 2, 3, 0)  # [C, H, W, 1]
    return image


def resize_and_crop_sr(image, image_size):
        # Crop slices in z dim
    center_idx = int(image.shape[0] / 2)
    num_slice = int(image_size[0] / 2)
    image = image[center_idx-num_slice:center_idx+num_slice, :, :]
    im_size = image.shape
    print(image.shape, center_idx, num_slice)

    # Complete 3D input image as a squared x-y image
    if not(im_size[1] == im_size[2]):
        length = np.max([im_size[1], im_size[2]])
        zerp_padding = np.zeros([im_size[0], length, length])
        if im_size[1] > im_size[2]:
            zerp_padding[:, :, np.int64((im_size[1]-im_size[2])/2):np.int64((im_size[1]-im_size[2])/2)+im_size[2]] = image
        else:
            zerp_padding[:, np.int64((im_size[2]-im_size[1])/2):np.int64((im_size[2]-im_size[1])/2)+im_size[1], :] = image
        image = zerp_padding
    # Resize image in x-y plane
    image = torch.tensor(image, dtype=torch.float32)[None, ...]  # [B, C, H, W]
    image_lr = F.interpolate(image, size=(image_size[1], image_size[2]), mode='bilinear', align_corners=False)

    # Scaling normalization
    image = image / torch.max(image)  # [B, C, H, W], [0, 1]
    # image = (image-image.min())/(image.max()-image.min())
    image = image.permute(1, 2, 3, 0)  # [C, H, W, 1]
    image_lr = image_lr / torch.max(image_lr)  # [B, C, H, W], [0, 1]
    image_lr = image_lr.permute(1, 2, 3, 0)  # [C, H, W, 1]
    return image, image_lr


def resize_volume(volume, size=(256, 256), save_root=None):
    volume = F.interpolate(torch.tensor(volume, dtype=torch.float32)[None, ...], size=size, mode='bilinear', align_corners=False)[0]
    volume = (volume - volume.min()) / (volume.max() - volume.min())
    volume = np.rot90(volume.numpy(), k=1, axes=(1, 2))
    if save_root is not None:
        if not os.path.exists(save_root):
            os.makedirs(save_root)
        for i in range(volume.shape[0]):
            image = volume[i]
            cv2.imwrite(os.path.join(save_root, f'{i:04d}.png'), (image*255).astype(np.uint8))
    return volume


def quick_evaluation(test_output,test_data):
    mse = torch.mean((test_output - test_data)**2)
    max_pixel_value = torch.max(test_data)
    if max_pixel_value != 1:
        print('max_pixel_value:', max_pixel_value)
    test_psnr = 10 * torch.log10(max_pixel_value / mse)
    test_ssim = compare_ssim(test_output.squeeze().cpu().numpy(), test_data.squeeze().cpu().numpy(), multichannel=True, data_range=2)
    print('MSE:{:.4f}, PSNR: {:.4f}, SSIM: {:.4f}'.format(mse*255*255/1000, test_psnr, test_ssim))
    sys.stdout.flush()

# def quick_evaluation(test_output,test_data):
#     loss_fn = torch.nn.MSELoss()
#     test_loss = 0.5 * loss_fn(test_output, test_data)
#     test_psnr = - 10 * torch.log10(2 * test_loss).item()
#     test_loss = test_loss.item()
#     test_ssim = compare_ssim(test_output.squeeze().cpu().numpy(), test_data.squeeze().cpu().numpy(), multichannel=True, data_range=2)
#     print('Test Loss: {:.4f}, PSNR: {:.4f}, SSIM: {:.4f}'.format(test_loss, test_psnr, test_ssim))

def count_neighbors(points, distance_threshold):
    tree = KDTree(points)
    neighbor_count = np.zeros(points.shape[0])
    for i in tqdm.tqdm(range(len(points))):
        num_neighbors = tree.query_radius(points[i].reshape(1,-1), distance_threshold, count_only=True)[0]
        neighbor_count[i] = num_neighbors - 1
    return neighbor_count

def count_neighbors_gpu(points, distance_threshold, batch_size=4096):
    points = torch.tensor(points).cuda()  # Move data to GPU
    distance_threshold = torch.tensor(distance_threshold).cuda()

    num_points = points.shape[0]
    neighbor_count = torch.zeros(num_points, device='cuda')

    # Process each batch separately
    for i in tqdm.tqdm(range(0, num_points, batch_size)):
        batch_points = points[i:i+batch_size]
        dist_matrix = torch.cdist(batch_points, points)  # Compute distances to all points
        neighbor_count[i:i+batch_size] = torch.sum(dist_matrix < distance_threshold, dim=1) - 1  # Count neighbors for each point in batch

    return neighbor_count.cpu().numpy()  # Move data back to CPU


def patched_fast_gaussian_computation(A, B, C, patch_size = 10000):
    _B = B[:,None,None,:]
    ATCA = []
    ATCB = []
    BTCA = []
    for i in range(0, B.shape[0], patch_size):
        ATCA.append(torch.einsum('ijkl,bll,ijkl->bijk',A,C[i: i+patch_size],A))
    ATCA = torch.cat(ATCA, dim=0)
    # del ATCA
    for i in range(0, B.shape[0], patch_size):
        ATCB.append(torch.einsum('ijkl,bll,bxxl->bijk',A,C[i: i+patch_size],_B[i:i+patch_size]))
    ATCB = torch.cat(ATCB, dim=0)
    # del ATCB
    for i in range(0, B.shape[0], patch_size):
        BTCA.append(torch.einsum('bxxl,bll,ijkl->bijk',_B[i: i+patch_size],C[i: i+patch_size],A))
    BTCA = torch.cat(BTCA, dim=0)
    # del BTCA
    BTCB =  torch.einsum('bxxl,bll,bxxl->b',_B,C,_B)[:,None,None,None]
    return ATCA - ATCB - BTCA + BTCB


def fast_gaussian_computation(A, B, C):
    """_summary_

    Args:
        A (_type_): impact_coords
        B (_type_): delta_impact_coords
        C (_type_): covariances inverse

    Returns:
        _type_: _description_
    """
    _B = B[:,None,None,:]
    ATCA = torch.einsum('ijkl,bll,ijkl->bijk',A,C,A)
    ATCB = torch.einsum('ijkl,bll,bxxl->bijk',A,C,_B)
    BTCA = torch.einsum('bxxl,bll,ijkl->bijk',_B,C,A)
    BTCB = torch.einsum('bxxl,bll,bxxl->b',_B,C,_B)
    return ATCA - ATCB - BTCA + BTCB[:,None,None,None]



if __name__ == '__main__':
    from odl.util.testutils import run_doctests
    run_doctests()
